### A Pluto.jl notebook ###
# v0.19.42

using Markdown
using InteractiveUtils

# This Pluto notebook uses @bind for interactivity. When running this notebook outside of Pluto, the following 'mock version' of @bind gives bound variables a default value (instead of an error).
macro bind(def, element)
    quote
        local iv = try Base.loaded_modules[Base.PkgId(Base.UUID("6e696c72-6542-2067-7265-42206c756150"), "AbstractPlutoDingetjes")].Bonds.initial_value catch; b -> missing; end
        local el = $(esc(element))
        global $(esc(def)) = Core.applicable(Base.get, el) ? Base.get(el) : iv(el)
        el
    end
end

# ╔═╡ 9069e664-d44e-4117-89df-c6193eee3835
begin
	using PlutoUI, HypertextLiteral, LinearAlgebra, StatsBase, PlotlyBase, PlutoPlotly , PlotlyKaleido
	force_pluto_mathjax_local(true)
	PlotlyKaleido.start(;mathjax = true)
end

# ╔═╡ 16a60e64-d7ff-4dd9-8042-d307b8725343
md"""
# Final test: Macroeconomics (M8674)

### Instituto Universitário de Lisboa (ISCTE--IUL)

**27 May 2024**
"""

# ╔═╡ 0d7a1da7-8183-4539-bad0-d17aff26a7cb
md"""
!!! danger "Compulsory"

	👉 Fill in your name and student number in the cells below.

"""

# ╔═╡ 925a7303-ece6-4d68-b9f4-df06b584b9f1
name = "My name"

# ╔═╡ 069febc9-84b8-464b-a866-a5f6a74bf635
number = 000000

# ╔═╡ 35cf15cc-d453-4d17-8759-8ac470e8b61e
begin
	function Base.show(io::IO, mimetype::MIME"text/html", plt::PlotlyBase.Plot)
       # Remove responsive flag on the plot as we handle responsibity via ResizeObeserver and leaving it on makes the div flickr during updates
	hasproperty(plt,:config) && plt.config.responsive && (plt.config.responsive = false)   
	show(io,mimetype, @htl("""
			<div>
			<script id=asdf>
			const {plotly} = await import("https://cdn.plot.ly/plotly-2.2.0.min.js")
			const PLOT = this ?? document.createElement("div");
		

		
			Plotly.react(PLOT, $(HypertextLiteral.JavaScript(PlotlyBase.json(plt))));


		
			const pluto_output = currentScript.parentElement.closest('pluto-output')

			const resizeObserver = new ResizeObserver(entries => {
				Plotly.Plots.resize(PLOT)
			})

			resizeObserver.observe(pluto_output)

			invalidation.then(() => {
				resizeObserver.disconnect()
			})
		
			return PLOT
			</script>
			</div>
	"""))
	end
end

# ╔═╡ 57eb2c01-7f16-46c0-a20e-ef0119e63c3a
md"""
## Packages
"""

# ╔═╡ f634151b-89e0-4b39-a6d3-7be5374cd239
md"""
## 1. The model
"""

# ╔═╡ e90e52a9-0e3f-472a-a70e-cbfb98c6b579
md"""
The model with endogenous government spending can be written as:

$$\begin{array}{lllll}
\text {IS : } & \hat{y}_{t}=\mathbb{E}_{t} \hat{y}_{t+1}-\frac{1}{\sigma}\left(i_{t}-\mathbb{E}_{t} \pi_{t+1}-r_{t}^{n}\right)\\
\text {Taylor rule : } & \ i_{t}  = \pi_t + r^n_t+\phi_{\pi} \pi_{t}+\phi_{y} \hat{y}_{t} + x_t \\
\text {AD : } & \hat{y}_{t}= \hat{c}_{t} \\
\text {AS : } & \pi_{t}  =\kappa \hat{y}_{t}+\beta \cdot \mathbb{E}_{t} \pi_{t+1} \\
\text {Real interest : } & r_{t}^{n}  =\rho_r \cdot r_{t-1}^{n}+\varepsilon_{t}^{r} \quad , \quad \varepsilon_{t}^{r} \sim {\cal{N}}(0,1) \\
\text {Decisions shock : } & x_{t}  =\rho_x \cdot x_{t-1}+\varepsilon_{t}^{x} \quad , \quad \varepsilon_{t}^{x} \sim {\cal{N}}(0,1) \\
\end{array}$$

- where $\{\hat{y}\ , \ i \ , \ \pi\ , \ r^n \ , \ \hat{c}\ , \ x_t \ , \ \varepsilon\}$ are, respectively, output gap, nominal interest rate, inflation rate, natural real interest rate, real consumption, decision-making process, and a random disturbance;
- while $\{ \beta \ , \ \sigma \ , \ \theta  \ , \ \phi_{y} \ , \ \phi_{\pi} \ , \ \rho_x \ , \ \rho_r \ , \ \psi  \ , \ \alpha \}$ 
"""

# ╔═╡ 5552d1b3-040a-43f8-8c14-07cd99d51ae6
md"""
The model can be written in state space form as:


$$\mathcal{A}\left[\begin{array}{c}
w_{t+1} \\
\mathbb{E}_{t} v_{t+1}
\end{array}\right]=\mathcal{B}\left[\begin{array}{c}
w_{t} \\
v_{t+1}
\end{array}\right]+\mathcal{C}\left[\begin{array}{c}
\varepsilon_{t+1}^{w} \\
\varepsilon_{t+1}^{v}
\end{array}\right]$$

where $w$ is a vector of backward-looking variables (it may include static vriables as well), and $v$ is a vector of forward looking variables. Multiplying both sides of the previous equation, leads to:

$$\left[\begin{array}{c}
w_{t+1} \\
\mathbb{E}_{t} v_{t+1}
\end{array}\right]=\underbrace{\mathcal{A}^{-1} \mathcal{B}}_{\mathcal{R}}\left[\begin{array}{c}
w_{t} \\
v_{t+1}
\end{array}\right]+\underbrace{\mathcal{A}^{-1} \mathcal{C}}_{\mathcal{U}}\left[\begin{array}{c}
\varepsilon_{t+1}^{w} \\
\varepsilon_{t+1}^{v}
\end{array}\right]$$
"""

# ╔═╡ 2431311d-d543-4c9f-a32e-15eef03564e8
md"""
$$\begin{array}{cccccccccc}
	      \beta & \sigma & \theta & \phi_{y}& \phi_{\pi} & \rho_x & \rho_r &\psi &  \alpha & r^n_t\\
	\hline 0.95 &   1.3  &   0.8  &    0.6  &     0.5    &   0.5  &  0.6 & 1.2 &   0.65  & 0
\end{array}$$

with $$\kappa=\frac{\psi(1-\theta)(1-\theta \beta)}{\theta}.$$
"""

# ╔═╡ 87d2e19a-c0e7-4d6c-812a-b3aa961ba029
md"""
## 2. Matrices
"""

# ╔═╡ c0df388e-49b3-4bf6-997e-0533c6aa4e6b
md"""
Parameters:
"""

# ╔═╡ f8a6084f-327f-4d9f-ad43-ddc95955cedf
begin
	β = 0.95 ;  σ = 1.3 ;  θ = 0.1 ;  ϕ_y = 0.6 ;  ϕ_π = 1.5 ;  ρₓ = 0.0 ;  ρᵣ = 0.6 ; ψ = 1.2 ;  α = 0.65 ; rⁿ = 0.0 ; 
	α = 0.6 ; κ = ψ*(1-θ)*(1-θ*β)/θ
end;

# ╔═╡ e66d90f1-b94d-491d-8837-d5fdee339315
md"""
Matrices:
"""

# ╔═╡ 425c783d-1f9b-4072-b93b-5c1e3e19f2be
begin
	A = zeros(3,3)
	B = zeros(3,3)
	C = zeros(3,3)
	
	A[1,1] =  1.0		
	A[2,2] =   β 	
	A[3,2] = 3.0/σ 
	A[3,3] = 1.0 
	
	B[1,1] =   ρₓ 
	B[2,2] =  0.0 
	B[2,3] = - κ
	B[3,1] =  1.0/σ
	B[3,2] = (1.0 + ϕ_π)/σ 
	B[3,3] = 1.0 + ϕ_y/σ 
		
	C[1,1] = 1.0
end;

# ╔═╡ 5493d224-cd8a-47ef-9413-232f8b2c3b87
A

# ╔═╡ 83fe1d2e-fd0c-41c6-81c0-867e168166f6
B

# ╔═╡ 87ed9b6f-cee7-44ba-8253-f1e134e94cbd
C

# ╔═╡ 7c089e4d-cdb7-4ea1-9f6f-093959529f1a
R = inv(A)*B

# ╔═╡ 1ff58a55-34aa-4efb-9ba4-d5199e4c84d2
U = inv(A)*C

# ╔═╡ ff2dae99-d0d4-47ee-a08c-5c1e9b8907c1
md"""
### Eigenvalues and eigenvectors
"""

# ╔═╡ 5f827a1d-beab-48cd-bd1e-73d168548517
eigen(R)

# ╔═╡ b2aa44ad-1ff8-4f86-bfcf-b89fa63aef9e
BK_condition = eigvals(R)

# ╔═╡ 225c0c47-b1b7-4cfb-b5f7-deaa4f1ae456
Λ = diagm(BK_condition)

# ╔═╡ f4510616-5f24-4b6f-a6f6-2b10750f5b43
eigenvectors = eigvecs(R)

# ╔═╡ cf2e2e15-ee32-4288-aa3f-baa54038b3d1
Pinv = inv(eigenvectors)

# ╔═╡ 26b6ec10-4d61-407f-a307-471aea5c90a7
md"""
## 3. The Blanchard-Kahn conditions
"""

# ╔═╡ 094673a0-362e-4545-9fe6-13e57f2e858a
md"""
In the cell below, insert the number of **_forward-looking_** (or jump) variables (`m`) and the number of **_non-forward-looking_** (predetermined and/or static) variables (`n`). Remember that everything in this notebook will depend upon setting the correct number of these two types of variables. All we have to do below is:

```julia
m = 2; # number of forward-looking (or jump) variables
n = 1; # number of backward-looking (or predetermined) variables
```
"""

# ╔═╡ 5f45a70b-33f5-44a7-aee0-32f513164497
begin
	m = 2 # number of forward-looking (or jump) variables
	n = 1 # number of backward-looking (or predetermined) variables
end;

# ╔═╡ 0f2b6197-c2ca-412e-a047-b3caf3c8effd
begin
	bk_n = 0;
	    for i = 1:length(BK_condition)
		    if abs.(BK_condition[i]) > 1.0
		        bk_n = bk_n+1;
		    end
	    end;
	println("BK condition");
	println("Number of jump variables:",m);
	println("Number of unstable roots:",bk_n);
end;

# ╔═╡ 8db65027-cc92-4aa0-8a84-84ff76268cb1
begin
	if bk_n == m;
	println("BK satisfied")
	elseif bk_n>n
	println("Too many unstable roots: stop")
	else
	println("Too few unstable roots: stop")
	end;
end;

# ╔═╡ 95c29eed-a1c8-4bea-941b-511d33c37923
md"""
As the Blanchard-Kahn conditions are satisfied, we can proceed with the simulation of this model. The next step is to implement the partition of the matrices $P^{-1}$ and $\Lambda$.
"""

# ╔═╡ 63e24427-e56f-48b6-90ae-5e163695ca45
md"""
## 4. Partition of matrices: $P^{-1} \ , \ \Lambda \ , \ \cal{M}$
"""

# ╔═╡ 5c6cbeeb-32e5-4a02-aecf-eff192f9f1dc
md"""
Before we do the partitions, a word of caution. It is crucial to partition the matrices above correctly. Otherwise, we may be simulating some models, but not the one we discussed.

The critical point is to separate the backward-looking block from the forward-looking one. As in the current model, we have two forward-looking variables and one non-forward-looking (a predetermined variable); we can organize the two blocks as it is presented in the yellow/cyan colors in the figure below. The eigenvalue less than $|1|$ should be the first element in the matrix's main diagonal $\Lambda$ (the yellow block), while the last two should be greater than $|1|$ (the cyan block). This logic should be matched by a similar partition on the $P^{-1}$ matrix on the right-hand side of such figure. 

A similar logic should be applied to the partition of matrix $\cal{M}=P^{-1} \mathcal{U}$, such that it includes only the shocks affecting the predetermined block $M{_{11}}$. For a better understanding of what we should do, the partition of the $\cal{M}$ matrix is displayed in the second image below.
"""

# ╔═╡ 93f951fa-2cd1-4ea3-b9b9-3e61f1c04362
Resource("http://ebs.de.iscte.pt/Partition_Matrices1x2_1.png", :width=>500)

# ╔═╡ c7165d31-983c-4571-8f0b-e5134fd48252
Resource("http://ebs.de.iscte.pt/Partition_Matrices1x2_2.png", :width=>500)

# ╔═╡ b97051fd-91cd-46b7-856c-3c0e42459328
md"""
Matrices partitions:
"""

# ╔═╡ 9403884b-55c7-43ca-86bf-d55d8b4371ab
begin
	Λ1  = Λ[1:n , 1:n] 	 					# rows 1; columns 1
	Λ2  = Λ[n+1:m+n , n+1:m+n]  			# rows 2,3; columns 2,3
	P11 = Pinv[1:n , 1:n] 					# rows 1; columns 1
	P12 = Pinv[1:n , n+1:m+n] 				# rows 1; columns 2,3
	P21 = Pinv[n+1:m+n , 1:n] 				# rows 2,3; columns 1
	P22 = Pinv[n+1:m+n , n+1:m+n] 			# rows 2,3; columns 2,3
	M   = real(Pinv*U) 						# Define matrix M
	M11 = M[1:n , 1:n] 						# rows 1; columns 1	
end;

# ╔═╡ d31eb439-c244-4bfd-a18f-eb15aea7f869
md"""
## 5. Solving the model
"""

# ╔═╡ 291a8c23-4721-48cd-9dcb-72bc559798ac
md"""
To solve the model, we have to apply the analytical solutions, first for the backward-looking block $(w_{t+1})$, then for the forward-looking one $(\mathbb {E}_t v_{t+1})$. For the former block, its solution is:

$$w_{t+1}^{*}=\underbrace{\left[G^{-1} \Lambda_{1} G\right]}_{g} \cdot w_{t}^{*}+\underbrace{\left[G^{-1} M_{11}\right]}_{h} \cdot \varepsilon_{t+1}$$

while for the forward-looking block, we have:

$$v_{t}^{*}=\underbrace{\left[-P_{22}^{-1} P_{21}\right]}_{f} \cdot w_{t}^{*}$$

with: $$\qquad \qquad \qquad \qquad \qquad \qquad \qquad \qquad G \equiv P_{11}-P_{12}\left(P_{22}\right)^{-1} P_{21}$$

Therefore, the only task is to compute the following matrices: $f,G,g,h.$ 
"""

# ╔═╡ 68af3f9b-85c6-4afc-ae6a-1ea576d61143
begin
	f = real(-inv(P22) * P21)
	G = real(P11 - P12 * inv(P22) * P21)
	g = real(inv(G) * (Λ1) * G)
	h = inv(G)*M11
end;

# ╔═╡ 85630efd-4bd0-4bd3-9e35-fd289bd4ebc1
md"""
## 6. Impulse response functions (IRFs)
"""

# ╔═╡ 6fe0e57a-c214-4a9b-bab4-53a00ed51917
md"""
Now, we can compute the solution to our model, using all the matrices caomputed in the previous section. We will use a `for loop` firtly to calculate the IRFs of the backward-looking block $w_t$, and then those of the forward-looking one $v_t$.
"""

# ╔═╡ 0a661a49-aa5d-46e6-a922-d8323140268d


# ╔═╡ 768609f0-84c5-4e1d-8420-67f48319055c
md"""
#### Labor and technology
##### (this section is also part of the model, but there is no time to explain it)
"""

# ╔═╡ 4ce1b592-dfc1-4a6e-bcac-2fe291419ad9
md"""
The model could easily be extended to include labor and technology. In the NKM usually, production is expressed without capital, so the output is just a function of technology $(a_t)$ and labor $(n_t)$:

$$y_t = a_t n^\alpha _t$$

with $\alpha$ as the parameter representing the share of labor in output. From this equation one can easily express it in terms of rates of change:

$$\hat{y}_t = \hat{a}_t + \alpha \hat{n}_t \ \Rightarrow \ \hat{a}_t = \hat{y}_t - \alpha \hat{n}_t$$

On the other hand, the optimal level of labor can be calculated as:

$$\hat{n}_t = \hat{y}_t - \sigma \hat{c}_t$$

Notice that these two results were not derivated in the entire structure of the model to avoid many small details that derail us from the fundamental points. However, it is not very complicated to obtain them.

"""

# ╔═╡ c0e847af-b23f-4fe3-b870-a366e5515c02
md"""
#### Summarizing the information from the simulation
"""

# ╔═╡ 059ac607-2aa5-4fb8-9454-6e9577e55aab
md"""
`decision_shock = `$(@bind decision_shock Slider(-1:1:1, default=1, show_value=true))........................# to remove the shock choose `0`
"""

# ╔═╡ 3b9298dc-daa2-422f-943e-b2a55b9864f6
begin
	T = 30 										# Number of iterations
	w_1 = zeros(n)								# Setting the inital conditions
	p_1 = length(w_1) 							# Passing the number of initial conditions
	ε = zeros(p_1, T)							# noise is a vector with 2 rows, t columns, all zeros
	ε[1,2] = decision_shock 						# the second element of row 1 is the shock
	w = [w_1  zeros(p_1, T-1)] 			    	# Pre-allocating memory 
	
		 for t = 1:T-1 		 					# the for loop for the non-forward-looking block
		     w[:, t+1] = g * w[:, t] + h * ε[:, t+1] # 
		 end
	
	v = f * w 									# Simulating the forward-looking block
end;

# ╔═╡ cfbae7a9-2b12-4c53-ae3c-37813d3fad74
begin
	shock =  w[1,:]
	infl  =  v[1,:]
	ygap  =  v[2,:]
	inter =  rⁿ .+ infl .+ ϕ_π .* infl .+  ϕ_y .* ygap
	labor =  ygap - σ * ygap
	tech  =  ygap - α * labor
end;
	

# ╔═╡ 6a260314-5c98-4961-9e15-bfde1ecc1283
begin
	trace_1 = scatter(;x = T, y = shock, 
				name = "s", mode="markers+lines",
				marker_symbol="circle", marker_size="7",line_width= 0.3, 
            	marker_color = "Blue")
	
	trace_2 = scatter(;x = T, y = ygap, 
				name = "y", mode="markers+lines",
				marker_symbol="circle", marker_size="7",line_width= 0.3,
            	marker_color = "Red")

	trace_3 = scatter(;x = T, y = infl, 
				name = "π", mode="markers+lines",
				marker_symbol="circle", marker_size="7",line_width= 0.3,
            	marker_color = "gray")

	trace_4 = scatter(;x = T, y = labor, 
				name = "n", mode="markers+lines",
				marker_symbol="circle", marker_size="7",line_width= 0.3,
            	marker_color = "darkred")

	trace_5 = scatter(;x = T, y = inter, 
				name = "i", mode="markers+lines",
				marker_symbol="circle", marker_size="7",line_width= 0.3,
            	marker_color = "orange")

	trace_6 = scatter(;x = T, y = tech, 
				name = "a", mode="markers+lines",
				marker_symbol="circle", marker_size="6",line_width= 0.3,
            	marker_color = "magenta")

	layout_4 = Layout(;
		
			title_text = "IRF: the NKM with a supply shock", title_x=0.5,
		
			hovermode="x",		
		
            xaxis = attr(
               title = "time",
               #tickformat = "%Y",
               #hoverformat = "%Y-M%m",
               #tick0 = "2003/01/01",
               #dtick = "M60" 
			),
		
        	#xaxis_range = [2000, 2020],
        	yaxis_title = "% points",
        	xaxis_range=[-0.4, 10.2], 
        	titlefont_size = 16)

	fig1 = Plot([trace_1, trace_2, trace_3, trace_4, trace_5, trace_6], layout_4)
end

# ╔═╡ 749d8787-2f8e-44d4-85b6-784c2acb11f8
md"""
## 7. Systematic shocks
"""

# ╔═╡ 81eeddd7-e07a-408e-bcce-7be4d265d0db
md"""
The only difference between an IRF and a time series is that, in the latter, the shocks hit that particular variable on every single period. So shocks happen all the time; they are systematic. Do not confuse this type of shock with what is framed as a "permanent shock" in economics. Using the example above, a permanent shock would mean that  $\varepsilon^s_{t+1}$ would remain equal to 1 all the time the simulation covers.

With the code for the IRFs already done, it is straightforward to write down the code for a systematic shock. We have to change the code in lines 4 and 5 above. Noise now, becomes `ε = randn(400)`, which means 400 white noise shocks in an array.

And, for a matter of clarification, we should change the names of the variables from "w" to just "w2" and "v" to "v2".
"""

# ╔═╡ ed247bf1-172c-441f-8e6d-18b3b1a5e283
begin
	T2 = 200 										# Number of iterations
	w2_1 = zeros(n)									# Setting the inital conditions
	p2 = length(w2_1) 								# Passing the number of initial conditions
	ε2 = randn(T2)' 							    # noise is a vector with 2 rows, t2 columns, all zeros	
	w2 = [w2_1 zeros(p2, T2-1)] 					# Pre-allocating memory
	
	for t = 1 : T2-1 								# the for loop for the stable block
		w2[:, t+1] = g * w2[:, t] + h * ε2[:, t+1] 
	end
	
	v2 = f * w2 									# Simulating the unstable block
end;

# ╔═╡ 83c44b8c-c142-4c05-9ad2-995a8dd55ebf
md"""
#### Defining the new variables
"""

# ╔═╡ beaff6e1-93e1-4ffc-8d44-643050c8d721
begin
	shock2  =  w2[1,:]
	infl2  	=  v2[1,:]
	ygap2  	=  v2[2,:]
	inter2 	=  rⁿ .+ infl2 .+ ϕ_π .* infl2 .+  ϕ_y .* ygap2
	labor2 	=  ygap2 - σ * ygap2
	tech2  	=  ygap2 - α * labor2
	variables2 = [shock2  infl2  ygap2  inter2  labor2  tech2]
end;

# ╔═╡ 853a6a45-78fe-4696-809a-2794de456d16
begin
		
	trace_14 = scatter(;x = T2, y = ygap2, 
				name = "y", mode="lines",
				marker_symbol="circle", marker_size="0",line_width= 1.5,
            	marker_color = "darkred")

	trace_15 = scatter(;x = T2, y = tech2, 
				name = "a", mode="lines",
				marker_symbol="triangle_up", marker_size="0",line_width= 1.5,
            	marker_color = "blueviolet")

	trace_16 = scatter(;x = T2, y = labor2, 
				name = "n", mode="lines",
				marker_symbol="triangle_down", marker_size="0", line_width= 1.5,
            	marker_color = "gray")

	layout_16 = Layout(;
		
			title_text = "Time series: the NKM with a supply shock", title_x=0.5,
		
			hovermode="x",		
		
            xaxis = attr(
               title = "time",
               #tickformat = "%Y",
               #hoverformat = "%Y-M%m",
               #tick0 = "2003/01/01",
               #dtick = "M60" 
			),
		
        	#xaxis_range = [2000, 2020],
        	yaxis_title = "% deviations",
        	#xaxis_range=[0, 200], 
        	titlefont_size = 16)

	fig2 = Plot([trace_14, trace_15, trace_16], layout_16)
end

# ╔═╡ b589f47e-9371-40e5-a265-8e70ca9785e0
begin
	trace_10 = scatter(;x = T2, y = shock2, 
				name = "s", mode="lines",
				marker_symbol="circle", marker_size="4",line_width= 1.5, 
            	marker_color = "red")
	
	trace_11 = scatter(;x = T2, y = ygap2, 
				name = "y", mode="lines",
				marker_symbol="circle", marker_size="4",line_width= 1.5,
            	marker_color = "black")

	trace_12 = scatter(;x = T2, y = infl2, 
				name = "π", mode="lines",
				marker_symbol="circle", marker_size="4",line_width= 1.5,
            	marker_color = "gray")

	trace_13 = scatter(;x = T2, y = inter2, 
				name = "i", mode="lines",
				marker_symbol="circle", marker_size="4",line_width= 1.5,
            	marker_color = "magenta")


	layout_13 = Layout(;
		
			title_text = "Time series: the NKM with a supply shock", title_x=0.5,
		
			hovermode="x",		
		
            xaxis = attr(
               title = "time",
               #tickformat = "%Y",
               #hoverformat = "%Y-M%m",
               #tick0 = "2003/01/01",
               #dtick = "M60" 
			),
		
        	#xaxis_range = [2000, 2020],
        	yaxis_title = "% deviations",
        	#xaxis_range=[0, 200], 
        	titlefont_size = 16)

	fig3 = Plot([trace_10, trace_11, trace_12, trace_13], layout_13)
end

# ╔═╡ 4a0a61e4-7f25-45f9-a81f-24228e38fbe0
md"""
## 8. Statistical measures
"""

# ╔═╡ 83790225-d8c0-4aa6-823a-b920b4fe1034
md"""
### Autocorrelation function
"""

# ╔═╡ f8a9eca1-ac81-414c-a23d-bdb3428aac8c
begin
	alags = 0:20
	auto_shock = autocor(shock2, alags)
	auto_infl = autocor(infl2, alags)
	auto_ygap = autocor(ygap2, alags)
	auto_inter = autocor(inter2, alags)
	auto_labor = autocor(labor2, alags)	
	auto_tech = autocor(tech2, alags)
		
	autocorrelation = [auto_shock  auto_infl  auto_ygap  auto_inter  auto_labor   auto_tech ]
end;

# ╔═╡ 02f2df36-cf06-4b02-b56b-983bf0d7b30a
begin
	fig4 = Plot(alags, autocorrelation, 
		mode="lines", line_width = "1.8", marker_size="6.5", marker_symbol = "circle",
		Layout(; title_text = "Autocorrelation function: the NKM with a supply shock", title_x = 0.5 , hovermode="x", xaxis_range=[0.0, 20.0]))
	restyle!(fig4, 1:6, name=["s", "π", "y", "i" , "n" , "a"])
	fig4
end

# ╔═╡ 4f5e82db-ed95-46be-8db1-b9ac5d05865b
md"""
### Cross-correlation function
"""

# ╔═╡ 69bf1be2-5282-4558-bc8d-5c51496ccf64
begin
	clags = -10:10;
	cross_y_y  = crosscor(ygap2, ygap2, clags)
	cross_y_i  = crosscor(ygap2, inter2, clags)
	cross_y_pi = crosscor(ygap2, infl2, clags)
	cross_y_g  = crosscor(ygap2, shock2, clags)
	cross_y_a  = crosscor(ygap2, tech2, clags)
	cross_y_n  = crosscor(ygap2, labor2, clags)
	crosscorrelation = [cross_y_y  cross_y_i cross_y_pi cross_y_g cross_y_a cross_y_n]
end;

# ╔═╡ 7fc9ecc5-4b03-42e1-9a8d-81ea9546282d
begin
	fig5 = Plot(clags, crosscorrelation, 
		mode="lines", line_width = "1.8", marker_size="6.5", marker_symbol = "circle",
		Layout(; title_text = "Cross-correlation function: the NKM with a supply shock", title_x = 0.5 , hovermode="x"))
	restyle!(fig5, 1:6, name=["s", "π", "y", "i" , "n" , "a"])
	fig5
end

# ╔═╡ fb37680c-37ad-449b-9d83-1518f09d176d
md"""
`lags = `$(@bind lags Slider(-10:1:10, default=0, show_value=true))
"""

# ╔═╡ 08a476b8-5d3b-4e5c-885a-0f2ed308d0a8
cross_corr_matrix = crosscor(variables2[:,:] , variables2[:,:], [lags]; demean=true);

# ╔═╡ 17cf21c1-d816-4aee-bf8f-58c243474703
heat_matrix = reshape(cross_corr_matrix, 6, 6)

# ╔═╡ 358d7acf-8185-4170-a5d0-cfd72ecc3e26
begin
	function heatmap1()
    trace = heatmap(
        x=["s", "π", "y", "i" , "n" , "a"],
        y=["s", "π", "y", "i" , "n" , "a"],
        z= heat_matrix, colorscale="bdu"
    				)
    Plot(trace, Layout(; title_text = "Heatmap: the NKM with a supply shock", title_x = 0.5 ))
	end
	fig6 = heatmap1()
end

# ╔═╡ 87d68066-a10a-46cc-8589-c9576e9f1930
#savefig(fig6, "hetamap.svg")

# ╔═╡ 41f0dce4-1103-4595-ab83-c1b11d4ddd2f
md"""
### Standard deviation
"""

# ╔═╡ 10dc0859-0aba-4609-9f1a-19b07c0512ac
begin
	sd_u  = std(shock2)
	sd_pi = std(infl2)
	sd_y  = std(ygap2)
	sd_i  = std(inter2)
	sd_n  = std(labor2)
	sd_a  = std(tech2)
	
	sd = [sd_u , sd_pi , sd_y , sd_i , sd_n , sd_a]
end

# ╔═╡ 6a3a9cc4-8c73-4521-8cb2-8ef629acf1bb

variables = ["Supply-Shock" , "Inflation" ,  "GDP" , "Interest" ,  "Labor" , "Technology"]


# ╔═╡ ae8750cb-60b3-4042-bbea-27a866c94146
begin
	trace_30 = bar(;x = variables, y = sd)
	
	layout_30 = Layout(;
		
			title_text = "Standard deviation in the NKM with a supply shock", title_x=0.5,
		
			hovermode="x",		
		
            xaxis = attr(
               #title = "time",
               #tickformat = "%Y",
               #hoverformat = "%Y-M%m",
               #tick0 = "2003/01/01",
               #dtick = "M60" 
			),
		
        	#xaxis_range = [2000, 2020],
        	yaxis_title = "Standard deviation",
        	#yaxis_range=[-2, 2], 
        	titlefont_size = 16)

	p30 = Plot([trace_30], layout_30)
end

# ╔═╡ 45655315-c237-40b6-90dc-bed0605ed684


# ╔═╡ 260a3eba-15e0-4a1e-8a0e-ea83983adf00
md"""
# Auxiliary cells (please, do not delete)
"""

# ╔═╡ b010543e-c8ca-4da1-a912-dde37a10e2ac
TableOfContents()

# ╔═╡ 10d809b8-8b58-4c5b-a1a4-263a13e487ef
html"""
<style>
	@media screen {
		main {
			margin: 0 auto;
			max-width: 1700px;
    		padding-left: max(200px, 10%);
    		padding-right: max(383px, 10%); 
            # 383px to accomodate TableOfContents(aside=true)
		}
	}
</style>
"""

# ╔═╡ 00000000-0000-0000-0000-000000000001
PLUTO_PROJECT_TOML_CONTENTS = """
[deps]
HypertextLiteral = "ac1192a8-f4b3-4bfe-ba22-af5b92cd3ab2"
LinearAlgebra = "37e2e46d-f89d-539d-b4ee-838fcccc9c8e"
PlotlyBase = "a03496cd-edff-5a9b-9e67-9cda94a718b5"
PlotlyKaleido = "f2990250-8cf9-495f-b13a-cce12b45703c"
PlutoPlotly = "8e989ff0-3d88-8e9f-f020-2b208a939ff0"
PlutoUI = "7f904dfe-b85e-4ff6-b463-dae2292396a8"
StatsBase = "2913bbd2-ae8a-5f71-8c99-4fb6c76f3a91"

[compat]
HypertextLiteral = "~0.9.3"
PlotlyBase = "~0.8.18"
PlotlyKaleido = "~2.2.4"
PlutoPlotly = "~0.3.6"
PlutoUI = "~0.7.37"
StatsBase = "~0.33.16"
"""

# ╔═╡ 00000000-0000-0000-0000-000000000002
PLUTO_MANIFEST_TOML_CONTENTS = """
# This file is machine-generated - editing it directly is not advised

julia_version = "1.10.0"
manifest_format = "2.0"
project_hash = "9e83a9c6f40cde2967b5d063b14e35667d7e0cda"

[[deps.AbstractPlutoDingetjes]]
deps = ["Pkg"]
git-tree-sha1 = "6e1d2a35f2f90a4bc7c2ed98079b2ba09c35b83a"
uuid = "6e696c72-6542-2067-7265-42206c756150"
version = "1.3.2"

[[deps.ArgTools]]
uuid = "0dad84c5-d112-42e6-8d28-ef12dabb789f"
version = "1.1.1"

[[deps.Artifacts]]
uuid = "56f22d72-fd6d-98f1-02f0-08ddc0907c33"

[[deps.Base64]]
uuid = "2a0f44e3-6c83-55bd-87e4-b1978d98bd5f"

[[deps.ColorSchemes]]
deps = ["ColorTypes", "ColorVectorSpace", "Colors", "FixedPointNumbers", "PrecompileTools", "Random"]
git-tree-sha1 = "4b270d6465eb21ae89b732182c20dc165f8bf9f2"
uuid = "35d6a980-a343-548e-a6ea-1d62b119f2f4"
version = "3.25.0"

[[deps.ColorTypes]]
deps = ["FixedPointNumbers", "Random"]
git-tree-sha1 = "b10d0b65641d57b8b4d5e234446582de5047050d"
uuid = "3da002f7-5984-5a60-b8a6-cbb66c0b333f"
version = "0.11.5"

[[deps.ColorVectorSpace]]
deps = ["ColorTypes", "FixedPointNumbers", "LinearAlgebra", "Requires", "Statistics", "TensorCore"]
git-tree-sha1 = "a1f44953f2382ebb937d60dafbe2deea4bd23249"
uuid = "c3611d14-8923-5661-9e6a-0046d554d3a4"
version = "0.10.0"

    [deps.ColorVectorSpace.extensions]
    SpecialFunctionsExt = "SpecialFunctions"

    [deps.ColorVectorSpace.weakdeps]
    SpecialFunctions = "276daf66-3868-5448-9aa4-cd146d93841b"

[[deps.Colors]]
deps = ["ColorTypes", "FixedPointNumbers", "Reexport"]
git-tree-sha1 = "362a287c3aa50601b0bc359053d5c2468f0e7ce0"
uuid = "5ae59095-9a9b-59fe-a467-6f913c188581"
version = "0.12.11"

[[deps.Compat]]
deps = ["TOML", "UUIDs"]
git-tree-sha1 = "b1c55339b7c6c350ee89f2c1604299660525b248"
uuid = "34da2185-b29b-5c13-b0c7-acf172513d20"
version = "4.15.0"
weakdeps = ["Dates", "LinearAlgebra"]

    [deps.Compat.extensions]
    CompatLinearAlgebraExt = "LinearAlgebra"

[[deps.CompilerSupportLibraries_jll]]
deps = ["Artifacts", "Libdl"]
uuid = "e66e0078-7015-5450-92f7-15fbd957f2ae"
version = "1.0.5+1"

[[deps.DataAPI]]
git-tree-sha1 = "abe83f3a2f1b857aac70ef8b269080af17764bbe"
uuid = "9a962f9c-6df0-11e9-0e5d-c546b8b5ee8a"
version = "1.16.0"

[[deps.DataStructures]]
deps = ["Compat", "InteractiveUtils", "OrderedCollections"]
git-tree-sha1 = "1d0a14036acb104d9e89698bd408f63ab58cdc82"
uuid = "864edb3b-99cc-5e75-8d2d-829cb0a9cfe8"
version = "0.18.20"

[[deps.Dates]]
deps = ["Printf"]
uuid = "ade2ca70-3891-5945-98fb-dc099432e06a"

[[deps.DelimitedFiles]]
deps = ["Mmap"]
git-tree-sha1 = "9e2f36d3c96a820c678f2f1f1782582fcf685bae"
uuid = "8bb1440f-4735-579b-a4ab-409b98df4dab"
version = "1.9.1"

[[deps.DocStringExtensions]]
deps = ["LibGit2"]
git-tree-sha1 = "2fb1e02f2b635d0845df5d7c167fec4dd739b00d"
uuid = "ffbed154-4ef7-542d-bbb7-c09d3a79fcae"
version = "0.9.3"

[[deps.Downloads]]
deps = ["ArgTools", "FileWatching", "LibCURL", "NetworkOptions"]
uuid = "f43a241f-c20a-4ad4-852c-f6b1247861c6"
version = "1.6.0"

[[deps.FileWatching]]
uuid = "7b1f6079-737a-58dc-b8bc-7a2ca5c1b5ee"

[[deps.FixedPointNumbers]]
deps = ["Statistics"]
git-tree-sha1 = "05882d6995ae5c12bb5f36dd2ed3f61c98cbb172"
uuid = "53c48c17-4a7d-5ca2-90c5-79b7896eea93"
version = "0.8.5"

[[deps.Hyperscript]]
deps = ["Test"]
git-tree-sha1 = "179267cfa5e712760cd43dcae385d7ea90cc25a4"
uuid = "47d2ed2b-36de-50cf-bf87-49c2cf4b8b91"
version = "0.0.5"

[[deps.HypertextLiteral]]
deps = ["Tricks"]
git-tree-sha1 = "7134810b1afce04bbc1045ca1985fbe81ce17653"
uuid = "ac1192a8-f4b3-4bfe-ba22-af5b92cd3ab2"
version = "0.9.5"

[[deps.IOCapture]]
deps = ["Logging", "Random"]
git-tree-sha1 = "8b72179abc660bfab5e28472e019392b97d0985c"
uuid = "b5f81e59-6552-4d32-b1f0-c071b021bf89"
version = "0.2.4"

[[deps.InteractiveUtils]]
deps = ["Markdown"]
uuid = "b77e0a4c-d291-57a0-90e8-8db25a27a240"

[[deps.IrrationalConstants]]
git-tree-sha1 = "630b497eafcc20001bba38a4651b327dcfc491d2"
uuid = "92d709cd-6900-40b7-9082-c6be49f344b6"
version = "0.2.2"

[[deps.JLLWrappers]]
deps = ["Artifacts", "Preferences"]
git-tree-sha1 = "7e5d6779a1e09a36db2a7b6cff50942a0a7d0fca"
uuid = "692b3bcd-3c85-4b1f-b108-f13ce0eb3210"
version = "1.5.0"

[[deps.JSON]]
deps = ["Dates", "Mmap", "Parsers", "Unicode"]
git-tree-sha1 = "31e996f0a15c7b280ba9f76636b3ff9e2ae58c9a"
uuid = "682c06a0-de6a-54ab-a142-c8b1cf79cde6"
version = "0.21.4"

[[deps.Kaleido_jll]]
deps = ["Artifacts", "JLLWrappers", "Libdl", "Pkg"]
git-tree-sha1 = "43032da5832754f58d14a91ffbe86d5f176acda9"
uuid = "f7e6163d-2fa5-5f23-b69c-1db539e41963"
version = "0.2.1+0"

[[deps.LaTeXStrings]]
git-tree-sha1 = "50901ebc375ed41dbf8058da26f9de442febbbec"
uuid = "b964fa9f-0449-5b57-a5c2-d3ea65f4040f"
version = "1.3.1"

[[deps.LibCURL]]
deps = ["LibCURL_jll", "MozillaCACerts_jll"]
uuid = "b27032c2-a3e7-50c8-80cd-2d36dbcbfd21"
version = "0.6.4"

[[deps.LibCURL_jll]]
deps = ["Artifacts", "LibSSH2_jll", "Libdl", "MbedTLS_jll", "Zlib_jll", "nghttp2_jll"]
uuid = "deac9b47-8bc7-5906-a0fe-35ac56dc84c0"
version = "8.4.0+0"

[[deps.LibGit2]]
deps = ["Base64", "LibGit2_jll", "NetworkOptions", "Printf", "SHA"]
uuid = "76f85450-5226-5b5a-8eaa-529ad045b433"

[[deps.LibGit2_jll]]
deps = ["Artifacts", "LibSSH2_jll", "Libdl", "MbedTLS_jll"]
uuid = "e37daf67-58a4-590a-8e99-b0245dd2ffc5"
version = "1.6.4+0"

[[deps.LibSSH2_jll]]
deps = ["Artifacts", "Libdl", "MbedTLS_jll"]
uuid = "29816b5a-b9ab-546f-933c-edad1886dfa8"
version = "1.11.0+1"

[[deps.Libdl]]
uuid = "8f399da3-3557-5675-b5ff-fb832c97cbdb"

[[deps.LinearAlgebra]]
deps = ["Libdl", "OpenBLAS_jll", "libblastrampoline_jll"]
uuid = "37e2e46d-f89d-539d-b4ee-838fcccc9c8e"

[[deps.LogExpFunctions]]
deps = ["DocStringExtensions", "IrrationalConstants", "LinearAlgebra"]
git-tree-sha1 = "18144f3e9cbe9b15b070288eef858f71b291ce37"
uuid = "2ab3a3ac-af41-5b50-aa03-7779005ae688"
version = "0.3.27"

    [deps.LogExpFunctions.extensions]
    LogExpFunctionsChainRulesCoreExt = "ChainRulesCore"
    LogExpFunctionsChangesOfVariablesExt = "ChangesOfVariables"
    LogExpFunctionsInverseFunctionsExt = "InverseFunctions"

    [deps.LogExpFunctions.weakdeps]
    ChainRulesCore = "d360d2e6-b24c-11e9-a2a3-2a2ae2dbcce4"
    ChangesOfVariables = "9e997f8a-9a97-42d5-a9f1-ce6bfc15e2c0"
    InverseFunctions = "3587e190-3f89-42d0-90ee-14403ec27112"

[[deps.Logging]]
uuid = "56ddb016-857b-54e1-b83d-db4d58db5568"

[[deps.MIMEs]]
git-tree-sha1 = "65f28ad4b594aebe22157d6fac869786a255b7eb"
uuid = "6c6e2e6c-3030-632d-7369-2d6c69616d65"
version = "0.1.4"

[[deps.Markdown]]
deps = ["Base64"]
uuid = "d6f4376e-aef5-505a-96c1-9c027394607a"

[[deps.MbedTLS_jll]]
deps = ["Artifacts", "Libdl"]
uuid = "c8ffd9c3-330d-5841-b78e-0817d7145fa1"
version = "2.28.2+1"

[[deps.Missings]]
deps = ["DataAPI"]
git-tree-sha1 = "ec4f7fbeab05d7747bdf98eb74d130a2a2ed298d"
uuid = "e1d29d7a-bbdc-5cf2-9ac0-f12de2c33e28"
version = "1.2.0"

[[deps.Mmap]]
uuid = "a63ad114-7e13-5084-954f-fe012c677804"

[[deps.MozillaCACerts_jll]]
uuid = "14a3606d-f60d-562e-9121-12d972cd8159"
version = "2023.1.10"

[[deps.NetworkOptions]]
uuid = "ca575930-c2e3-43a9-ace4-1e988b2c1908"
version = "1.2.0"

[[deps.OpenBLAS_jll]]
deps = ["Artifacts", "CompilerSupportLibraries_jll", "Libdl"]
uuid = "4536629a-c528-5b80-bd46-f80d51c5b363"
version = "0.3.23+2"

[[deps.OrderedCollections]]
git-tree-sha1 = "dfdf5519f235516220579f949664f1bf44e741c5"
uuid = "bac558e1-5e72-5ebc-8fee-abe8a469f55d"
version = "1.6.3"

[[deps.PackageExtensionCompat]]
git-tree-sha1 = "fb28e33b8a95c4cee25ce296c817d89cc2e53518"
uuid = "65ce6f38-6b18-4e1d-a461-8949797d7930"
version = "1.0.2"
weakdeps = ["Requires", "TOML"]

[[deps.Parameters]]
deps = ["OrderedCollections", "UnPack"]
git-tree-sha1 = "34c0e9ad262e5f7fc75b10a9952ca7692cfc5fbe"
uuid = "d96e819e-fc66-5662-9728-84c9c7592b0a"
version = "0.12.3"

[[deps.Parsers]]
deps = ["Dates", "PrecompileTools", "UUIDs"]
git-tree-sha1 = "8489905bcdbcfac64d1daa51ca07c0d8f0283821"
uuid = "69de0a69-1ddd-5017-9359-2bf0b02dc9f0"
version = "2.8.1"

[[deps.Pkg]]
deps = ["Artifacts", "Dates", "Downloads", "FileWatching", "LibGit2", "Libdl", "Logging", "Markdown", "Printf", "REPL", "Random", "SHA", "Serialization", "TOML", "Tar", "UUIDs", "p7zip_jll"]
uuid = "44cfe95a-1eb2-52ea-b672-e2afdf69b78f"
version = "1.10.0"

[[deps.PlotlyBase]]
deps = ["ColorSchemes", "Dates", "DelimitedFiles", "DocStringExtensions", "JSON", "LaTeXStrings", "Logging", "Parameters", "Pkg", "REPL", "Requires", "Statistics", "UUIDs"]
git-tree-sha1 = "56baf69781fc5e61607c3e46227ab17f7040ffa2"
uuid = "a03496cd-edff-5a9b-9e67-9cda94a718b5"
version = "0.8.19"

[[deps.PlotlyKaleido]]
deps = ["Base64", "JSON", "Kaleido_jll"]
git-tree-sha1 = "2650cd8fb83f73394996d507b3411a7316f6f184"
uuid = "f2990250-8cf9-495f-b13a-cce12b45703c"
version = "2.2.4"

[[deps.PlutoPlotly]]
deps = ["AbstractPlutoDingetjes", "Colors", "Dates", "HypertextLiteral", "InteractiveUtils", "LaTeXStrings", "Markdown", "PackageExtensionCompat", "PlotlyBase", "PlutoUI", "Reexport"]
git-tree-sha1 = "9a77654cdb96e8c8a0f1e56a053235a739d453fe"
uuid = "8e989ff0-3d88-8e9f-f020-2b208a939ff0"
version = "0.3.9"
weakdeps = ["PlotlyKaleido"]

    [deps.PlutoPlotly.extensions]
    PlotlyKaleidoExt = "PlotlyKaleido"

[[deps.PlutoUI]]
deps = ["AbstractPlutoDingetjes", "Base64", "ColorTypes", "Dates", "FixedPointNumbers", "Hyperscript", "HypertextLiteral", "IOCapture", "InteractiveUtils", "JSON", "Logging", "MIMEs", "Markdown", "Random", "Reexport", "URIs", "UUIDs"]
git-tree-sha1 = "ab55ee1510ad2af0ff674dbcced5e94921f867a9"
uuid = "7f904dfe-b85e-4ff6-b463-dae2292396a8"
version = "0.7.59"

[[deps.PrecompileTools]]
deps = ["Preferences"]
git-tree-sha1 = "5aa36f7049a63a1528fe8f7c3f2113413ffd4e1f"
uuid = "aea7be01-6a6a-4083-8856-8a6e6704d82a"
version = "1.2.1"

[[deps.Preferences]]
deps = ["TOML"]
git-tree-sha1 = "9306f6085165d270f7e3db02af26a400d580f5c6"
uuid = "21216c6a-2e73-6563-6e65-726566657250"
version = "1.4.3"

[[deps.Printf]]
deps = ["Unicode"]
uuid = "de0858da-6303-5e67-8744-51eddeeeb8d7"

[[deps.REPL]]
deps = ["InteractiveUtils", "Markdown", "Sockets", "Unicode"]
uuid = "3fa0cd96-eef1-5676-8a61-b3b8758bbffb"

[[deps.Random]]
deps = ["SHA"]
uuid = "9a3f8284-a2c9-5f02-9a11-845980a1fd5c"

[[deps.Reexport]]
git-tree-sha1 = "45e428421666073eab6f2da5c9d310d99bb12f9b"
uuid = "189a3867-3050-52da-a836-e630ba90ab69"
version = "1.2.2"

[[deps.Requires]]
deps = ["UUIDs"]
git-tree-sha1 = "838a3a4188e2ded87a4f9f184b4b0d78a1e91cb7"
uuid = "ae029012-a4dd-5104-9daa-d747884805df"
version = "1.3.0"

[[deps.SHA]]
uuid = "ea8e919c-243c-51af-8825-aaa63cd721ce"
version = "0.7.0"

[[deps.Serialization]]
uuid = "9e88b42a-f829-5b0c-bbe9-9e923198166b"

[[deps.Sockets]]
uuid = "6462fe0b-24de-5631-8697-dd941f90decc"

[[deps.SortingAlgorithms]]
deps = ["DataStructures"]
git-tree-sha1 = "66e0a8e672a0bdfca2c3f5937efb8538b9ddc085"
uuid = "a2af1166-a08f-5f64-846c-94a0d3cef48c"
version = "1.2.1"

[[deps.SparseArrays]]
deps = ["Libdl", "LinearAlgebra", "Random", "Serialization", "SuiteSparse_jll"]
uuid = "2f01184e-e22b-5df5-ae63-d93ebab69eaf"
version = "1.10.0"

[[deps.Statistics]]
deps = ["LinearAlgebra", "SparseArrays"]
uuid = "10745b16-79ce-11e8-11f9-7d13ad32a3b2"
version = "1.10.0"

[[deps.StatsAPI]]
deps = ["LinearAlgebra"]
git-tree-sha1 = "1ff449ad350c9c4cbc756624d6f8a8c3ef56d3ed"
uuid = "82ae8749-77ed-4fe6-ae5f-f523153014b0"
version = "1.7.0"

[[deps.StatsBase]]
deps = ["DataAPI", "DataStructures", "LinearAlgebra", "LogExpFunctions", "Missings", "Printf", "Random", "SortingAlgorithms", "SparseArrays", "Statistics", "StatsAPI"]
git-tree-sha1 = "d1bf48bfcc554a3761a133fe3a9bb01488e06916"
uuid = "2913bbd2-ae8a-5f71-8c99-4fb6c76f3a91"
version = "0.33.21"

[[deps.SuiteSparse_jll]]
deps = ["Artifacts", "Libdl", "libblastrampoline_jll"]
uuid = "bea87d4a-7f5b-5778-9afe-8cc45184846c"
version = "7.2.1+1"

[[deps.TOML]]
deps = ["Dates"]
uuid = "fa267f1f-6049-4f14-aa54-33bafae1ed76"
version = "1.0.3"

[[deps.Tar]]
deps = ["ArgTools", "SHA"]
uuid = "a4e569a6-e804-4fa4-b0f3-eef7a1d5b13e"
version = "1.10.0"

[[deps.TensorCore]]
deps = ["LinearAlgebra"]
git-tree-sha1 = "1feb45f88d133a655e001435632f019a9a1bcdb6"
uuid = "62fd8b95-f654-4bbd-a8a5-9c27f68ccd50"
version = "0.1.1"

[[deps.Test]]
deps = ["InteractiveUtils", "Logging", "Random", "Serialization"]
uuid = "8dfed614-e22c-5e08-85e1-65c5234f0b40"

[[deps.Tricks]]
git-tree-sha1 = "eae1bb484cd63b36999ee58be2de6c178105112f"
uuid = "410a4b4d-49e4-4fbc-ab6d-cb71b17b3775"
version = "0.1.8"

[[deps.URIs]]
git-tree-sha1 = "67db6cc7b3821e19ebe75791a9dd19c9b1188f2b"
uuid = "5c2747f8-b7ea-4ff2-ba2e-563bfd36b1d4"
version = "1.5.1"

[[deps.UUIDs]]
deps = ["Random", "SHA"]
uuid = "cf7118a7-6976-5b1a-9a39-7adc72f591a4"

[[deps.UnPack]]
git-tree-sha1 = "387c1f73762231e86e0c9c5443ce3b4a0a9a0c2b"
uuid = "3a884ed6-31ef-47d7-9d2a-63182c4928ed"
version = "1.0.2"

[[deps.Unicode]]
uuid = "4ec0a83e-493e-50e2-b9ac-8f72acf5a8f5"

[[deps.Zlib_jll]]
deps = ["Libdl"]
uuid = "83775a58-1f1d-513f-b197-d71354ab007a"
version = "1.2.13+1"

[[deps.libblastrampoline_jll]]
deps = ["Artifacts", "Libdl"]
uuid = "8e850b90-86db-534c-a0d3-1478176c7d93"
version = "5.8.0+1"

[[deps.nghttp2_jll]]
deps = ["Artifacts", "Libdl"]
uuid = "8e850ede-7688-5339-a07c-302acd2aaf8d"
version = "1.52.0+1"

[[deps.p7zip_jll]]
deps = ["Artifacts", "Libdl"]
uuid = "3f19e933-33d8-53b3-aaab-bd5110c3b7a0"
version = "17.4.0+2"
"""

# ╔═╡ Cell order:
# ╟─16a60e64-d7ff-4dd9-8042-d307b8725343
# ╟─0d7a1da7-8183-4539-bad0-d17aff26a7cb
# ╠═925a7303-ece6-4d68-b9f4-df06b584b9f1
# ╠═069febc9-84b8-464b-a866-a5f6a74bf635
# ╟─35cf15cc-d453-4d17-8759-8ac470e8b61e
# ╟─57eb2c01-7f16-46c0-a20e-ef0119e63c3a
# ╠═9069e664-d44e-4117-89df-c6193eee3835
# ╟─f634151b-89e0-4b39-a6d3-7be5374cd239
# ╟─e90e52a9-0e3f-472a-a70e-cbfb98c6b579
# ╟─5552d1b3-040a-43f8-8c14-07cd99d51ae6
# ╟─2431311d-d543-4c9f-a32e-15eef03564e8
# ╟─87d2e19a-c0e7-4d6c-812a-b3aa961ba029
# ╟─c0df388e-49b3-4bf6-997e-0533c6aa4e6b
# ╠═f8a6084f-327f-4d9f-ad43-ddc95955cedf
# ╟─e66d90f1-b94d-491d-8837-d5fdee339315
# ╠═425c783d-1f9b-4072-b93b-5c1e3e19f2be
# ╠═5493d224-cd8a-47ef-9413-232f8b2c3b87
# ╠═83fe1d2e-fd0c-41c6-81c0-867e168166f6
# ╠═87ed9b6f-cee7-44ba-8253-f1e134e94cbd
# ╠═7c089e4d-cdb7-4ea1-9f6f-093959529f1a
# ╠═1ff58a55-34aa-4efb-9ba4-d5199e4c84d2
# ╟─ff2dae99-d0d4-47ee-a08c-5c1e9b8907c1
# ╠═5f827a1d-beab-48cd-bd1e-73d168548517
# ╠═b2aa44ad-1ff8-4f86-bfcf-b89fa63aef9e
# ╠═225c0c47-b1b7-4cfb-b5f7-deaa4f1ae456
# ╠═f4510616-5f24-4b6f-a6f6-2b10750f5b43
# ╠═cf2e2e15-ee32-4288-aa3f-baa54038b3d1
# ╟─26b6ec10-4d61-407f-a307-471aea5c90a7
# ╟─094673a0-362e-4545-9fe6-13e57f2e858a
# ╠═5f45a70b-33f5-44a7-aee0-32f513164497
# ╟─0f2b6197-c2ca-412e-a047-b3caf3c8effd
# ╟─8db65027-cc92-4aa0-8a84-84ff76268cb1
# ╟─95c29eed-a1c8-4bea-941b-511d33c37923
# ╟─63e24427-e56f-48b6-90ae-5e163695ca45
# ╟─5c6cbeeb-32e5-4a02-aecf-eff192f9f1dc
# ╟─93f951fa-2cd1-4ea3-b9b9-3e61f1c04362
# ╟─c7165d31-983c-4571-8f0b-e5134fd48252
# ╟─b97051fd-91cd-46b7-856c-3c0e42459328
# ╠═9403884b-55c7-43ca-86bf-d55d8b4371ab
# ╟─d31eb439-c244-4bfd-a18f-eb15aea7f869
# ╟─291a8c23-4721-48cd-9dcb-72bc559798ac
# ╠═68af3f9b-85c6-4afc-ae6a-1ea576d61143
# ╟─85630efd-4bd0-4bd3-9e35-fd289bd4ebc1
# ╟─6fe0e57a-c214-4a9b-bab4-53a00ed51917
# ╠═3b9298dc-daa2-422f-943e-b2a55b9864f6
# ╠═0a661a49-aa5d-46e6-a922-d8323140268d
# ╟─768609f0-84c5-4e1d-8420-67f48319055c
# ╟─4ce1b592-dfc1-4a6e-bcac-2fe291419ad9
# ╟─c0e847af-b23f-4fe3-b870-a366e5515c02
# ╠═cfbae7a9-2b12-4c53-ae3c-37813d3fad74
# ╠═059ac607-2aa5-4fb8-9454-6e9577e55aab
# ╟─6a260314-5c98-4961-9e15-bfde1ecc1283
# ╟─749d8787-2f8e-44d4-85b6-784c2acb11f8
# ╟─81eeddd7-e07a-408e-bcce-7be4d265d0db
# ╠═ed247bf1-172c-441f-8e6d-18b3b1a5e283
# ╟─83c44b8c-c142-4c05-9ad2-995a8dd55ebf
# ╠═beaff6e1-93e1-4ffc-8d44-643050c8d721
# ╟─853a6a45-78fe-4696-809a-2794de456d16
# ╟─b589f47e-9371-40e5-a265-8e70ca9785e0
# ╟─4a0a61e4-7f25-45f9-a81f-24228e38fbe0
# ╟─83790225-d8c0-4aa6-823a-b920b4fe1034
# ╠═f8a9eca1-ac81-414c-a23d-bdb3428aac8c
# ╟─02f2df36-cf06-4b02-b56b-983bf0d7b30a
# ╠═4f5e82db-ed95-46be-8db1-b9ac5d05865b
# ╠═69bf1be2-5282-4558-bc8d-5c51496ccf64
# ╠═7fc9ecc5-4b03-42e1-9a8d-81ea9546282d
# ╟─fb37680c-37ad-449b-9d83-1518f09d176d
# ╠═08a476b8-5d3b-4e5c-885a-0f2ed308d0a8
# ╠═17cf21c1-d816-4aee-bf8f-58c243474703
# ╟─358d7acf-8185-4170-a5d0-cfd72ecc3e26
# ╠═87d68066-a10a-46cc-8589-c9576e9f1930
# ╟─41f0dce4-1103-4595-ab83-c1b11d4ddd2f
# ╠═10dc0859-0aba-4609-9f1a-19b07c0512ac
# ╠═6a3a9cc4-8c73-4521-8cb2-8ef629acf1bb
# ╟─ae8750cb-60b3-4042-bbea-27a866c94146
# ╠═45655315-c237-40b6-90dc-bed0605ed684
# ╟─260a3eba-15e0-4a1e-8a0e-ea83983adf00
# ╠═b010543e-c8ca-4da1-a912-dde37a10e2ac
# ╟─10d809b8-8b58-4c5b-a1a4-263a13e487ef
# ╟─00000000-0000-0000-0000-000000000001
# ╟─00000000-0000-0000-0000-000000000002
